# @copyright (c) 2002-2014 Acronis International GmbH. All rights reserved.

from .translations import get_functions
from jinja2.utils import LRUCache

import datetime
import locale
import time
import jinja2


STRING_PREFIX = '<from string>:'


__all__ = ['get_loader', 'clear_loader_cache']


__loaders_cache = LRUCache(10)


def format_number_local(number):
    if not isinstance(number, (int, float)):
        return 'N/A'
    if isinstance(number, float):
        return locale.format('%.3f', number, grouping=True)
    return locale.format('%d', number, grouping=True)


def format_currency_local(value):
    if not isinstance(value, (int, float)):
        return 'N/A'
    return locale.currency(value, grouping=True)


class _Template:
    def __init__(self, template, translation, localizations=None):
        self.__impl = template
        self.__translation = translation
        self.__localizations = localizations

    def substitute(self, **kwargs):
        if self.__translation:
            kwargs.update(get_functions(self.__translation))
        if self.__localizations:
            kwargs.update(self.__localizations)
        return self.__impl.render(kwargs)

    def render_to_file(self, values, file_obj):
        """
        Render template to file directly. Similar to self.substitute, but renders the template piece by piece
        and immediately dumps each portion to the file. This allows it to be used for generating very large files.
        """
        if self.__translation:
            values.update(get_functions(self.__translation))
        if self.__localizations:
            values.update(self.__localizations)

        # We can't use self.__impl.stream(values).dump(file_obj)
        # because of the strange behavior of codecs.StreamWriter.writelines:
        # it dumps all data at the same time instead of line by line
        for text_block in self.__impl.stream(values):
            file_obj.write(text_block)


class _LoaderFromString(jinja2.BaseLoader):
    def get_source(self, environment, template):
        if not isinstance(template, str) or not template.startswith(STRING_PREFIX):
            raise jinja2.TemplateNotFound(template)
        return template[len(STRING_PREFIX):], None, lambda: True


class _Loader(jinja2.Environment):
    default_language = 'en_US'

    def __init__(self, search_path, translations=None, globals=None):
        loader = jinja2.ChoiceLoader([
            _LoaderFromString(),
            jinja2.FileSystemLoader(search_path)
        ])

        super().__init__(extensions=['jinja2.ext.i18n'], loader=loader, auto_reload=False)
        self.system_locale_only = globals.pop('system_locale_only', False) if globals else False
        self.install_null_translations()
        self.globals.update(datetime=datetime)
        self.globals.update(globals or {})
        self.__translations = translations or {}

    def load(self, name, language=None):
        language = language or self.default_language
        translation = self.__translations.get(language) or \
                      self.__translations.get(language.replace('-', '_')) or \
                      self.__translations.get(language.split('-')[0])
        return _Template(super().get_template(name), translation,
                         self.get_localization_functions(language))

    def get_localization_functions(self, language):
        if self.system_locale_only:
            return {
                'format_number': format_number_local,
                'format_currency': format_currency_local,
                'format_date': lambda x: time.strftime('%x', x.timetuple()),
                'format_date_full': lambda x: time.strftime('%c', x.timetuple())
            }
        else:
            from babel.numbers import format_decimal, format_currency
            from babel.dates import format_datetime, format_date
            babel_language = language.replace('-','_')
            return {
                'format_number': lambda x: format_decimal(x, '0.00', locale=babel_language),
                'format_currency': lambda x, cur: format_currency(x, cur, locale=babel_language),
                'format_date':  lambda x: format_date(x, format='medium', locale=babel_language),
                'format_date_full':  lambda x: format_datetime(x, format='medium', locale=babel_language)
            }


def get_loader(search_path, translations=None, globals=None):
    key = search_path if isinstance(search_path, str) else tuple(search_path)

    loader = __loaders_cache.get(key)
    if loader is None:
        __loaders_cache[key] = loader = _Loader(search_path, translations, globals)
    return loader


def clear_loader_cache():
    __loaders_cache.clear()
