import acrobind.protection
from . backup_set import *
from . constants import *
from . slices import *
from . import logs
import datetime
import acrort
import logging


def get_backup_comment(backup_set):
    names = {
        BACKUP_SET_DAILY: 'Daily backup',
        BACKUP_SET_WEEKLY: 'Weekly backup',
        BACKUP_SET_MONTHLY: 'Monthly backup',
        BACKUP_SET_HOURLY: 'Hourly backup',
        BACKUP_SET_FULL: 'Full backup',
        BACKUP_SET_INC: 'Incremental backup',
        BACKUP_SET_DIFF: 'Differential backup'
    }
    return names[backup_set]


def calculate_custom_backup_set(backup_type):
    backup_sets = {
        acrobind.protection.BACKUP_TYPE_FULL: BACKUP_SET_FULL,
        acrobind.protection.BACKUP_TYPE_INCREMENTAL: BACKUP_SET_INC,
        acrobind.protection.BACKUP_TYPE_DIFFERENTIAL: BACKUP_SET_DIFF,
        acrobind.protection.BACKUP_TYPE_INCREMENTAL_AS_SYNTHETIC_FULL: BACKUP_SET_FULL
    }
    if not backup_type in backup_sets:
        acrort.common.make_logic_error('backup_type={} cannot be matched to backup set properly.'.format(backup_type)).throw()
    logging.info("[calculate_next_backup_type_command][calculate_custom_backup_set] Backup type: {}".format(backup_sets[backup_type]))
    return backup_sets[backup_type]


def calculate_next_backup_set(service, backup_plan):
    scheme_type = backup_plan.Scheme.Type.ref
    schedule_type = get_schedule_type(backup_plan)
    week_start = get_week_beginning(backup_plan)
    backup_set_calculator = create_backup_set_calculator(schedule_type, week_start)
    target_stage = make_stage_spec(backup_plan, backup_plan.Route.Stages[0])
    previous_slice = select_last_slice(service, target_stage, backup_plan['Options'])
    backup_time = datetime.datetime.now(datetime.timezone.utc).astimezone()
    next_slice = {'id': None, 'kind': None, 'creation_time': backup_time, 'uri': ''}
    logging.info("[calculate_next_backup_type_command][calculate_next_backup_set] Previous slice: {},\nnext slice: {}".format(previous_slice, next_slice))
    return backup_set_calculator(next_slice, previous_slice)


def get_backup_type(scheme_type, backup_set):
    weekly_full_daily_incr_backup_sets = {
        BACKUP_SET_WEEKLY: acrobind.protection.BACKUP_TYPE_FULL,
        BACKUP_SET_DAILY: acrobind.protection.BACKUP_TYPE_INCREMENTAL
    }
    monthly_full_weekly_diff_daily_incr_backup_sets = {
        BACKUP_SET_MONTHLY: acrobind.protection.BACKUP_TYPE_FULL,
        BACKUP_SET_WEEKLY: acrobind.protection.BACKUP_TYPE_DIFFERENTIAL,
        BACKUP_SET_DAILY: acrobind.protection.BACKUP_TYPE_INCREMENTAL
    }
    scheme_mapping = {
        acrobind.protection.BACKUP_SCHEME_ALWAYS_FULL: (lambda: acrobind.protection.BACKUP_TYPE_FULL),
        acrobind.protection.BACKUP_SCHEME_ALWAYS_INCREMENTAL: (lambda: acrobind.protection.BACKUP_TYPE_INCREMENTAL),
        acrobind.protection.BACKUP_SCHEME_WEEKLY_FULL_DAILY_INCR: (lambda: weekly_full_daily_incr_backup_sets[backup_set]),
        acrobind.protection.BACKUP_SCHEME_MONTHLY_FULL_WEEKLY_DIFF_DAILY_INCR: (lambda: monthly_full_weekly_diff_daily_incr_backup_sets[backup_set])
    }
    caclulate_backup_type = scheme_mapping[scheme_type]
    return caclulate_backup_type()


def calculate_backup_type_command(service, backup_plan):
    scheme_type = backup_plan.Scheme.Type.ref
    try:
        if scheme_type in [acrobind.protection.BACKUP_SCHEME_CUSTOM, acrobind.protection.BACKUP_SCHEME_SIMPLE]:
            next_backup_set = calculate_custom_backup_set(backup_plan.Type.ref)
            backup_type = backup_plan.Type.ref
        else:
            next_backup_set = calculate_next_backup_set(service, backup_plan)
            backup_type = get_backup_type(scheme_type, next_backup_set)
    except SliceInFutureError as err:
        SliceInFutureErrorToLogicError(err, backup_plan.Route.Stages[0]).throw()
    backup_comment = get_backup_comment(next_backup_set)
    logging.info("[calculate_next_backup_type_command][calculate_backup_type_command] Summary: %s", {
        'BackupType': ('dword', backup_type),
        'BackupSet': ('dword', next_backup_set),
        'BackupComment': backup_comment
    })
    return acrort.plain.Unit({
        'BackupType': ('dword', backup_type),
        'BackupSet': ('dword', next_backup_set),
        'BackupComment': backup_comment
    })
