# @copyright (c) 2002-2013 Acronis International GmbH. All rights reserved.

import threading
import weakref


class _IterationGuard:
    def __init__(self, container):
        self.weak_container = weakref.ref(container)

    def __enter__(self):
        w = self.weak_container()
        if w is not None:
            w.add_iterator(self)
        return self

    def __exit__(self, *_):
        w = self.weak_container()
        if w is not None:
            w.remove_iterator(self)
        return False


class SharedDict:
    def __init__(self, *args, **kwargs):
        self.main = dict(*args, **kwargs)
        self._adds = dict()
        self._removals = set()
        self._iterators = set()
        self.lock = threading.Lock()

    def add_iterator(self, it):
        with self.lock:
            self._iterators.add(it)

    def remove_iterator(self, it):
        with self.lock:
            self._iterators.remove(it)
            if not self._iterators:
                self._commit()

    def _commit(self):
        while self._adds:
            key, value = self._adds.popitem()
            self.main[key] = value

        while self._removals:
            try:
                del self.main[self._removals.pop()]
            except KeyError:
                pass

    def __getitem__(self, key):
        return self.main[key]

    def __setitem__(self, key, value):
        with self.lock:
            if key in self.main or not self._iterators:
                self.main[key] = value
            else:
                self._adds[key] = value

    def __delitem__(self, key):
        with self.lock:
            if self._iterators:
                self._removals.add(key)
            else:
                del self.main[key]

    def __contains__(self, key):
        return key in self.main

    def __bool__(self):
        return bool(self.main)

    def __len__(self):
        return len(self.main)

    def get(self, key, default=None):
        return self.main.get(key, default)

    def pop(self, key, default=None):
        with self.lock:
            if self._iterators:
                self._removals.add(key)
                return self.main.get(key, default)
            return self.main.pop(key, default)

    def __iter__(self):
        return self.keys()

    def items(self):
        with _IterationGuard(self):
            for x, y in self.main.items():
                yield (x, y)

    def keys(self):
        with _IterationGuard(self):
            for x in self.main.keys():
                yield x

    def values(self):
        with _IterationGuard(self):
            for x in self.main.values():
                yield x
