# @copyright (c) 2002-2013 Acronis International GmbH. All rights reserved.

from . import config as _config
from . import logging
import builtins
import os
import signal
import stat
import sys

try:
    from acronis.debug import debug
except ImportError:
    debug = None


def _daemonize():
    try:
        pid = os.fork()
        if pid != 0:
            sys.exit(0)  # exit parent process

    except OSError as e:
        sys.exit('Could not become a daemon: fork #1 failed: {!r}'.format(e))

    os.setsid()

    try:
        pid = os.fork()
        if pid != 0:
            sys.exit(0)  # exit session leader
    except OSError as e:
        sys.exit('Could not become a daemon: fork #21 failed: {!r}'.format(e))

    os.umask(stat.S_IWOTH)  # disable: S_IWOTH
    os.chdir('/')

    sys.stdin = open(os.devnull, 'a+')
    sys.stdout = open(os.devnull, 'w')
    sys.stderr = open(os.devnull, 'w')


def _write_pid(filepath):
    with open(filepath, 'w') as pid_file:
        pid_file.write(str(os.getpid()))


def daemon(application_class):
    from argparse import ArgumentParser

    parser = ArgumentParser(application_class.__doc__)
    parser.add_argument('-c', '--config', dest='config', help='configuration file', required=True)
    parser.add_argument('--debug', action='store_false', dest='daemon', help='debug mode', default=True)
    args, _ = parser.parse_known_args()
    logger = None
    try:
        print('config file:', args.config)
        builtins.appconfig = _config.load_from_file(args.config, vars={'name': os.path.basename(sys.argv[0])})
        pid_file = appconfig.get('daemon', {}).get('pidfile')
        args.daemon and _daemonize()
        pid_file and _write_pid(pid_file)
        logger = logging.create_multi_process_logger(**appconfig['logging'])
        application = application_class(appconfig)
        signal.signal(signal.SIGTERM, application.terminate)
        signal.signal(signal.SIGINT, application.terminate)
        signal.signal(signal.SIGUSR1, logger.rotate)
        debug and signal.signal(signal.SIGUSR2, debug.enable_remote_console())
        logger.start()
        application.start()
        application.join()
        logging.info('exited.')
        sys.exit(0)
    except Exception as e:
        import traceback
        traceback.print_exc()
        logging.exception('{!r} crashed: {!r}', application_class, e)
        sys.exit(1)
    finally:
        logger and logger.stop()
