import acrort
import os
import string
import sys

from . cmdline_parser import ArgumentsProcessor, ArgumentParser
from . output import Output


TAG_CMD_SHELL = 'cmd-shell'
TAG_STOP_ON_ERROR = 'stop-on-error'
TAG_CMD_SCRIPT = 'cmd-script'


def format_docstring(docstring):
    lines = docstring.expandtabs().splitlines()
    indent = sys.maxsize
    for line in lines[1:]:
        stripped = line.lstrip()
        if stripped:
            indent = min(indent, len(line) - len(stripped))
    trimmed = [lines[0].strip()]
    if indent < sys.maxsize:
        for line in lines[1:]:
            trimmed.append(line[indent:].rstrip())
    while trimmed and not trimmed[-1]:
        trimmed.pop()
    while trimmed and not trimmed[0]:
        trimmed.pop(0)
    return '\n'.join(trimmed)


class CommandShellArgumentsProcessor(ArgumentsProcessor):
    def create_arguments_parser(self):
        parser = ArgumentParser(add_help=False)
        parser.add_argument('--cmd-script', nargs=1, metavar=('SCRIPT'))
        parser.add_argument('--stop-on-error', action='store_true')
        return parser

    def process_arguments(self, config, args):
        output = {}
        output[TAG_STOP_ON_ERROR] = args.stop_on_error
        if args.cmd_script is not None:
            output[TAG_CMD_SCRIPT] = args.cmd_script[0]
        config[TAG_CMD_SHELL] = output


class CommandShell:
    def __init__(self, *, connection, printer=None, config=None):
        self._prompt = '> '
        self._identchars = string.ascii_letters + string.digits + '-'
        self.connection = connection
        if printer is None:
            printer = Output(end='n')
        self.printer = printer
        self._stop_on_error = False
        self._cmd_script = None

        if config is not None:
            config = config.get(TAG_CMD_SHELL)
        if config is not None:
            self._stop_on_error = config.get(TAG_STOP_ON_ERROR, self._stop_on_error)
            self._cmd_script = config.get(TAG_CMD_SCRIPT)

    def preloop(self):
        pass

    def cmdloop(self):
        self.preloop()
        ret = None
        if self._cmd_script is not None:
            with open(self._cmd_script, 'rt') as cmdfile:
                ret = self.cmdloop_for_file(cmdfile)
        if ret is not None:
            return ret
        self.printer.write('')
        self.printer.write("Type command 'help' for details or command 'exit' to finish.", end='\n')
        ret = self.cmdloop_for_file(sys.stdin)
        if ret is None:
            ret = acrort.common.INTERRUPT_AWARE_RETURN_CODE
        return ret

    def cmdloop_for_file(self, cmdfile):
        is_interactive = os.isatty(cmdfile.fileno())
        if is_interactive:
            self.printer.write('\n', self._prompt, end='')

        ret = None
        for line in cmdfile:
            line = line.rstrip('\r\n').strip()
            if line.startswith('#'):
                line = None
            if line:
                if not is_interactive:
                    self.printer.write('\n', self._prompt, end='')
                echo = False if is_interactive else True
                self.printer.write(line, echo=echo, end='\n')
            if self.connection.sentinel:
                self.printer.write(self.connection.status)
                ret = acrort.common.EXCEPTION_AWARE_RETURN_CODE
                break
            try:
                stop = None
                if line:
                    stop = self._exec_cmd(line)
                if stop:
                    ret = 0
                    break
            except:
                error = acrort.common.get_current_exception_as_error()
                self.printer.write(error)
                if self._stop_on_error:
                    ret = acrort.common.EXCEPTION_AWARE_RETURN_CODE
                    break

            if is_interactive:
                self.printer.write('\n', self._prompt, end='')
        return ret

    def _exec_cmd(self, line):
        i, n = 0, len(line)
        while i < n and line[i] in self._identchars:
            i = i + 1
        cmd, arg = line[:i], line[i:].strip()
        handler_name = 'do_' + cmd.replace('-', '_')
        try:
            handler = getattr(self, handler_name)
        except AttributeError:
            if not cmd:
                cmd = line
            msg = "Unknown command: '{}' is given.".format(cmd)
            if self._stop_on_error:
                acrort.common.make_logic_error(msg).throw()
            else:
                self.printer.write(msg)
                return
        return handler(arg)

    def do_exit(self, arg):
        """Stop commands execution and exit."""
        return True

    def do_doc(self, arg):
        """List documentation for all available commands."""
        self.do_help(None)

    def do_help(self, arg):
        """List available commands with "help" or detailed help with "help cmd"."""
        names = dir(self.__class__)
        names.sort()
        help = {}
        help_nodoc = set()
        prevname = ''
        for name in names:
            if name[:3] != 'do_':
                continue
            if name == prevname:  # there can be duplicates if routines overridden
                continue
            prevname = name
            cmd = name[3:].replace('_', '-')
            cmd_doc = getattr(self, name).__doc__
            if cmd_doc:
                help[cmd] = format_docstring(cmd_doc)
            else:
                help_nodoc.add(cmd)
        if not arg:
            doc_title = 'Documented commands (type help <topic>):'
            self.printer.write(doc_title, end='\n')
            if arg is None:
                self.printer.write(len(doc_title) * '-', end='\n')
            for name in sorted(help.keys()):
                self.printer.write(": ", name, end='\n')
                cmd_doc = help[name]
                if arg is None:
                    self.printer.write(cmd_doc, end='\n\n')
            if help_nodoc and (arg is not None):
                self.printer.write('\n', 'Undocumented commands:', end='\n')
                for name in sorted(help_nodoc):
                    self.printer.write(": ", name, end='\n')
        else:
            cmd_doc = help.get(arg)
            if cmd_doc is None:
                self.printer.write("No help for command: {}".format(arg), end='\n')
                return
            self.printer.write(cmd_doc, end='\n')
